import { Injectable } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';

import { msg } from '@lingui/core/macro';
import { type ObjectsPermissionsByRoleId } from 'twenty-shared/types';
import { isDefined } from 'twenty-shared/utils';
import { In, Repository } from 'typeorm';

import { RelationType } from 'src/engine/metadata-modules/field-metadata/interfaces/relation-type.interface';

import {
  InternalServerError,
  UserInputError,
} from 'src/engine/core-modules/graphql/utils/graphql-errors.util';
import { FieldMetadataEntity } from 'src/engine/metadata-modules/field-metadata/field-metadata.entity';
import { isFieldMetadataTypeRelation } from 'src/engine/metadata-modules/field-metadata/utils/is-field-metadata-type-relation.util';
import { WorkspaceManyOrAllFlatEntityMapsCacheService } from 'src/engine/metadata-modules/flat-entity/services/workspace-many-or-all-flat-entity-maps-cache.service';
import { findFlatEntityByIdInFlatEntityMaps } from 'src/engine/metadata-modules/flat-entity/utils/find-flat-entity-by-id-in-flat-entity-maps.util';
import { type FlatEntityMaps } from 'src/engine/metadata-modules/flat-entity/types/flat-entity-maps.type';
import { type FlatFieldMetadata } from 'src/engine/metadata-modules/flat-field-metadata/types/flat-field-metadata.type';
import { type FlatObjectMetadata } from 'src/engine/metadata-modules/flat-object-metadata/types/flat-object-metadata.type';
import { type UpsertFieldPermissionsInput } from 'src/engine/metadata-modules/object-permission/dtos/upsert-field-permissions.input';
import { FieldPermissionEntity } from 'src/engine/metadata-modules/object-permission/field-permission/field-permission.entity';
import {
  PermissionsException,
  PermissionsExceptionCode,
  PermissionsExceptionMessage,
} from 'src/engine/metadata-modules/permissions/permissions.exception';
import { RoleEntity } from 'src/engine/metadata-modules/role/role.entity';
import { WorkspaceCacheService } from 'src/engine/workspace-cache/services/workspace-cache.service';

@Injectable()
export class FieldPermissionService {
  constructor(
    @InjectRepository(RoleEntity)
    private readonly roleRepository: Repository<RoleEntity>,
    @InjectRepository(FieldMetadataEntity)
    private readonly fieldMetadataRepository: Repository<FieldMetadataEntity>,
    @InjectRepository(FieldPermissionEntity)
    private readonly fieldPermissionsRepository: Repository<FieldPermissionEntity>,
    private readonly workspaceCacheService: WorkspaceCacheService,
    private readonly workspaceManyOrAllFlatEntityMapsCacheService: WorkspaceManyOrAllFlatEntityMapsCacheService,
  ) {}

  public async upsertFieldPermissions({
    workspaceId,
    input,
  }: {
    workspaceId: string;
    input: UpsertFieldPermissionsInput;
  }): Promise<FieldPermissionEntity[]> {
    const role = await this.getRoleOrThrow({
      roleId: input.roleId,
      workspaceId,
    });

    const { rolesPermissions } =
      await this.workspaceCacheService.getOrRecompute(workspaceId, [
        'rolesPermissions',
      ]);

    await this.validateRoleIsEditableOrThrow({
      role,
    });

    const { flatObjectMetadataMaps, flatFieldMetadataMaps } =
      await this.workspaceManyOrAllFlatEntityMapsCacheService.getOrRecomputeManyOrAllFlatEntityMaps(
        {
          workspaceId,
          flatMapsKeys: ['flatObjectMetadataMaps', 'flatFieldMetadataMaps'],
        },
      );

    const existingFieldPermissions = await this.fieldPermissionsRepository.find(
      {
        where: {
          roleId: input.roleId,
          workspaceId,
        },
      },
    );

    const fieldPermissionsToDeleteIds: string[] = [];

    input.fieldPermissions.forEach((fieldPermission) => {
      this.validateFieldPermission({
        allFieldPermissions: input.fieldPermissions,
        fieldPermission,
        flatObjectMetadataMaps,
        flatFieldMetadataMaps,
        rolesPermissions,
        role,
      });

      if (
        fieldPermission.canReadFieldValue === null ||
        fieldPermission.canUpdateFieldValue === null
      ) {
        this.checkIfFieldPermissionShouldBeDeleted({
          fieldPermission,
          existingFieldPermissions,
          fieldPermissionsToDeleteIds,
        });
      }
    });

    const fieldPermissions = input.fieldPermissions.map((fieldPermission) => ({
      ...fieldPermission,
      roleId: input.roleId,
      workspaceId,
    }));

    const existingFieldPermissionsToDelete = existingFieldPermissions.filter(
      (existingFieldPermissionToFilter) =>
        fieldPermissionsToDeleteIds.includes(
          existingFieldPermissionToFilter.id,
        ),
    );

    const fieldPermissionsToUpsert = fieldPermissions.filter(
      (fieldPermissionToUpsert) =>
        !existingFieldPermissionsToDelete.some(
          (existingFieldPermissionToDelete) =>
            existingFieldPermissionToDelete.fieldMetadataId ===
            fieldPermissionToUpsert.fieldMetadataId,
        ),
    );

    const fieldMetadatasForFieldPermissions =
      await this.fieldMetadataRepository.find({
        where: {
          id: In(fieldPermissions.map((fp) => fp.fieldMetadataId)),
        },
      });

    const relatedFieldPermissionsToUpsert =
      this.computeFieldPermissionForRelationTargetFieldMetadata({
        fieldPermissions: fieldPermissionsToUpsert,
        fieldMetadatasForFieldPermissions,
      });

    await this.fieldPermissionsRepository.upsert(
      [...fieldPermissionsToUpsert, ...relatedFieldPermissionsToUpsert],
      {
        conflictPaths: ['fieldMetadataId', 'roleId'],
      },
    );

    if (fieldPermissionsToDeleteIds.length > 0) {
      const relatedFieldPermissionToDeleteIds =
        this.getRelatedFieldPermissionsToDeleteIds({
          allFieldPermissions: existingFieldPermissions,
          fieldPermissionsToDelete: existingFieldPermissionsToDelete,
          fieldMetadatas: fieldMetadatasForFieldPermissions,
        });

      await this.fieldPermissionsRepository.delete({
        id: In([
          ...fieldPermissionsToDeleteIds,
          ...relatedFieldPermissionToDeleteIds,
        ]),
      });
    }

    await this.workspaceCacheService.invalidateAndRecompute(workspaceId, [
      'rolesPermissions',
    ]);

    return this.fieldPermissionsRepository.find({
      where: {
        roleId: input.roleId,
        objectMetadataId: In(
          input.fieldPermissions.map(
            (fieldPermission) => fieldPermission.objectMetadataId,
          ),
        ),
        workspaceId,
      },
    });
  }

  private validateFieldPermission({
    allFieldPermissions,
    fieldPermission,
    flatObjectMetadataMaps,
    flatFieldMetadataMaps,
    rolesPermissions,
    role,
  }: {
    allFieldPermissions: UpsertFieldPermissionsInput['fieldPermissions'];
    fieldPermission: UpsertFieldPermissionsInput['fieldPermissions'][0];
    flatObjectMetadataMaps: FlatEntityMaps<FlatObjectMetadata>;
    flatFieldMetadataMaps: FlatEntityMaps<FlatFieldMetadata>;
    rolesPermissions: ObjectsPermissionsByRoleId;
    role: RoleEntity;
  }) {
    const duplicateFieldPermissions = allFieldPermissions.filter(
      (permission) =>
        permission.fieldMetadataId === fieldPermission.fieldMetadataId,
    );

    if (duplicateFieldPermissions.length > 1) {
      throw new UserInputError(
        `Cannot accept more than one fieldPermission for field ${fieldPermission.fieldMetadataId} in input.`,
      );
    }
    if (
      ('canUpdateFieldValue' in fieldPermission &&
        fieldPermission.canUpdateFieldValue !== null &&
        fieldPermission.canUpdateFieldValue !== false) ||
      ('canReadFieldValue' in fieldPermission &&
        fieldPermission.canReadFieldValue !== null &&
        fieldPermission.canReadFieldValue !== false)
    ) {
      throw new PermissionsException(
        PermissionsExceptionMessage.ONLY_FIELD_RESTRICTION_ALLOWED,
        PermissionsExceptionCode.ONLY_FIELD_RESTRICTION_ALLOWED,
        {
          userFriendlyMessage: msg`Field permissions can only be used to restrict access, not to grant additional permissions.`,
        },
      );
    }

    const flatObjectMetadata = findFlatEntityByIdInFlatEntityMaps({
      flatEntityId: fieldPermission.objectMetadataId,
      flatEntityMaps: flatObjectMetadataMaps,
    });

    if (!isDefined(flatObjectMetadata)) {
      throw new PermissionsException(
        PermissionsExceptionMessage.OBJECT_METADATA_NOT_FOUND,
        PermissionsExceptionCode.OBJECT_METADATA_NOT_FOUND,
        {
          userFriendlyMessage: msg`The object you are trying to set permissions for could not be found. It may have been deleted.`,
        },
      );
    }

    if (flatObjectMetadata.isSystem === true) {
      throw new PermissionsException(
        PermissionsExceptionMessage.CANNOT_ADD_FIELD_PERMISSION_ON_SYSTEM_OBJECT,
        PermissionsExceptionCode.CANNOT_ADD_FIELD_PERMISSION_ON_SYSTEM_OBJECT,
        {
          userFriendlyMessage: msg`You cannot set field permissions on system objects as they are managed by the platform.`,
        },
      );
    }

    const flatFieldMetadata = findFlatEntityByIdInFlatEntityMaps({
      flatEntityId: fieldPermission.fieldMetadataId,
      flatEntityMaps: flatFieldMetadataMaps,
    });

    if (!isDefined(flatFieldMetadata)) {
      throw new PermissionsException(
        PermissionsExceptionMessage.FIELD_METADATA_NOT_FOUND,
        PermissionsExceptionCode.FIELD_METADATA_NOT_FOUND,
        {
          userFriendlyMessage: msg`The field you are trying to set permissions for could not be found. It may have been deleted.`,
        },
      );
    }

    const rolePermissionOnObject =
      rolesPermissions?.[role.id]?.[fieldPermission.objectMetadataId];

    if (!isDefined(rolePermissionOnObject)) {
      throw new PermissionsException(
        PermissionsExceptionMessage.OBJECT_PERMISSION_NOT_FOUND,
        PermissionsExceptionCode.OBJECT_PERMISSION_NOT_FOUND,
        {
          userFriendlyMessage: msg`No permissions are set for this role on the selected object. Please set object permissions first.`,
        },
      );
    }
  }

  private async getRoleOrThrow({
    roleId,
    workspaceId,
  }: {
    roleId: string;
    workspaceId: string;
  }) {
    const role = await this.roleRepository.findOne({
      where: {
        id: roleId,
        workspaceId,
      },
      relations: ['objectPermissions', 'fieldPermissions'],
    });

    if (!isDefined(role)) {
      throw new PermissionsException(
        PermissionsExceptionMessage.ROLE_NOT_FOUND,
        PermissionsExceptionCode.ROLE_NOT_FOUND,
        {
          userFriendlyMessage: msg`The role you are trying to modify could not be found. It may have been deleted or you may not have access to it.`,
        },
      );
    }

    return role;
  }

  private async validateRoleIsEditableOrThrow({ role }: { role: RoleEntity }) {
    if (!role.isEditable) {
      throw new PermissionsException(
        PermissionsExceptionMessage.ROLE_NOT_EDITABLE,
        PermissionsExceptionCode.ROLE_NOT_EDITABLE,
        {
          userFriendlyMessage: msg`This role cannot be modified because it is a system role. Only custom roles can be edited.`,
        },
      );
    }
  }

  private checkIfFieldPermissionShouldBeDeleted({
    fieldPermission,
    existingFieldPermissions,
    fieldPermissionsToDeleteIds,
  }: {
    fieldPermission: UpsertFieldPermissionsInput['fieldPermissions'][0];
    existingFieldPermissions: FieldPermissionEntity[];
    fieldPermissionsToDeleteIds: string[];
  }) {
    const existingFieldPermission = existingFieldPermissions.find(
      (existingFieldPermission) =>
        existingFieldPermission.fieldMetadataId ===
        fieldPermission.fieldMetadataId,
    );

    if (existingFieldPermission) {
      const finalCanReadFieldValue =
        'canReadFieldValue' in fieldPermission
          ? fieldPermission.canReadFieldValue
          : existingFieldPermission.canReadFieldValue;
      const finalCanUpdateFieldValue =
        'canUpdateFieldValue' in fieldPermission
          ? fieldPermission.canUpdateFieldValue
          : existingFieldPermission.canUpdateFieldValue;

      if (
        finalCanReadFieldValue === null &&
        finalCanUpdateFieldValue === null
      ) {
        fieldPermissionsToDeleteIds.push(existingFieldPermission.id);
      }
    }
  }

  private getRelatedFieldPermissionsToDeleteIds({
    allFieldPermissions,
    fieldPermissionsToDelete,
    fieldMetadatas,
  }: {
    allFieldPermissions: FieldPermissionEntity[];
    fieldPermissionsToDelete: FieldPermissionEntity[];
    fieldMetadatas: FieldMetadataEntity[];
  }) {
    const fieldMetadatasForFieldPermissionsToDelete = fieldMetadatas.filter(
      (fieldMetadata) =>
        fieldPermissionsToDelete.some(
          (existingFieldPermissionToDelete) =>
            existingFieldPermissionToDelete.fieldMetadataId ===
            fieldMetadata.id,
        ),
    );

    const relationTargetFieldMetadataIds: string[] = [];

    for (const fieldMetadataForFieldPermissionToDelete of fieldMetadatasForFieldPermissionsToDelete) {
      if (
        isFieldMetadataTypeRelation(fieldMetadataForFieldPermissionToDelete)
      ) {
        if (
          fieldMetadataForFieldPermissionToDelete.settings?.relationType ===
            RelationType.ONE_TO_MANY ||
          fieldMetadataForFieldPermissionToDelete.settings?.relationType ===
            RelationType.MANY_TO_ONE
        ) {
          relationTargetFieldMetadataIds.push(
            fieldMetadataForFieldPermissionToDelete.relationTargetFieldMetadataId,
          );
        }
      }
    }

    const fieldPermissionsForRelationTargetFieldMetadataIds =
      allFieldPermissions
        .filter((fieldPermission) =>
          relationTargetFieldMetadataIds.includes(
            fieldPermission.fieldMetadataId,
          ),
        )
        .map((fieldPermission) => fieldPermission.id);

    return fieldPermissionsForRelationTargetFieldMetadataIds;
  }

  private computeFieldPermissionForRelationTargetFieldMetadata({
    fieldPermissions,
    fieldMetadatasForFieldPermissions,
  }: {
    fieldPermissions: UpsertFieldPermissionsInput['fieldPermissions'];
    fieldMetadatasForFieldPermissions: FieldMetadataEntity[];
  }) {
    return fieldPermissions
      .map((fieldPermission) => {
        const fieldMetadata = fieldMetadatasForFieldPermissions.find(
          (fm) => fm.id === fieldPermission.fieldMetadataId,
        );

        if (!isDefined(fieldMetadata)) {
          throw new InternalServerError(
            'Field metadata not found for field permission',
          );
        }

        if (isFieldMetadataTypeRelation(fieldMetadata)) {
          if (
            fieldMetadata.settings?.relationType === RelationType.ONE_TO_MANY ||
            fieldMetadata.settings?.relationType === RelationType.MANY_TO_ONE
          ) {
            const fieldPermissionsOnRelationTargetField =
              fieldPermissions.filter(
                (fieldPermissionInput) =>
                  fieldPermissionInput.fieldMetadataId ===
                  fieldMetadata.relationTargetFieldMetadataId,
              );

            if (fieldPermissionsOnRelationTargetField.length > 0) {
              const firstFieldPermission =
                fieldPermissionsOnRelationTargetField[0]; // validation rules guarantee there can only be one

              const hasConflictingPermissions =
                fieldPermission.canReadFieldValue !==
                  firstFieldPermission.canReadFieldValue ||
                fieldPermission.canUpdateFieldValue !==
                  firstFieldPermission.canUpdateFieldValue;

              if (hasConflictingPermissions) {
                const fieldName = fieldMetadata.name;

                throw new UserInputError(
                  'Conflicting field permissions found for relation target field',
                  {
                    userFriendlyMessage: msg`Contradicting field permissions have been detected on a relation field (${fieldName}).`,
                  },
                );
              }

              return;
            }

            return {
              ...fieldPermission,
              objectMetadataId: fieldMetadata.relationTargetObjectMetadataId,
              fieldMetadataId: fieldMetadata.relationTargetFieldMetadataId,
            };
          }
        }

        return null;
      })
      .filter(isDefined);
  }
}
