import { useRecoilCallback } from 'recoil';

import { filterRecordOnGqlFields } from '@/object-record/cache/utils/filterRecordOnGqlFields';
import { type RecordGqlFields } from '@/object-record/graphql/record-gql-fields/types/RecordGqlFields';
import { recordStoreFamilyState } from '@/object-record/record-store/states/recordStoreFamilyState';
import { recordStoreFamilyStateV2 } from '@/object-record/record-store/states/recordStoreFamilyStateV2';
import { type ObjectRecord } from '@/object-record/types/ObjectRecord';
import { useStore } from 'jotai';
import { isDefined } from 'twenty-shared/utils';
import { isDeeplyEqual } from '~/utils/isDeeplyEqual';

type UpsertRecordsInStoreProps = {
  partialRecords: ObjectRecord[];
  recordGqlFields?: RecordGqlFields;
};

export const useUpsertRecordsInStore = () => {
  const store = useStore();

  const upsertRecordsInStore = useRecoilCallback(
    ({ set, snapshot }) =>
      ({ partialRecords, recordGqlFields }: UpsertRecordsInStoreProps) => {
        for (const partialRecord of partialRecords) {
          const currentRecord = snapshot
            .getLoadable(recordStoreFamilyState(partialRecord.id))
            .getValue();

          const filteredPartialRecord = isDefined(recordGqlFields)
            ? filterRecordOnGqlFields({
                record: partialRecord,
                recordGqlFields,
              })
            : partialRecord;

          if (!isDefined(currentRecord)) {
            const newRecord = {
              id: partialRecord.id,
              __typename: partialRecord.__typename,
              ...filteredPartialRecord,
            };
            set(recordStoreFamilyState(partialRecord.id), newRecord);
            store.set(
              recordStoreFamilyStateV2.atomFamily(partialRecord.id),
              newRecord,
            );
            continue;
          }

          const filteredCurrentRecord = isDefined(recordGqlFields)
            ? filterRecordOnGqlFields({
                record: currentRecord,
                recordGqlFields,
              })
            : currentRecord;

          if (!isDeeplyEqual(filteredCurrentRecord, filteredPartialRecord)) {
            const updatedRecord = {
              ...currentRecord,
              ...filteredPartialRecord,
            };
            set(recordStoreFamilyState(partialRecord.id), updatedRecord);
            store.set(
              recordStoreFamilyStateV2.atomFamily(partialRecord.id),
              updatedRecord,
            );
          }
        }
      },
    [store],
  );

  return {
    upsertRecordsInStore,
  };
};
