import fs from 'fs';
import path from 'path';

type BasePage = string | BaseGroup;

type BaseGroup = {
  key: string;
  label: string;
  icon?: string;
  pages: BasePage[];
};

type BaseTab = {
  key: string;
  label: string;
  groups: BaseGroup[];
};

type BaseStructure = {
  tabs: BaseTab[];
};

const baseStructurePath = path.resolve(
  __dirname,
  '../navigation/base-structure.json',
);

const outputDir = path.resolve(__dirname, '../../twenty-shared/src/constants');

const baseStructure: BaseStructure = JSON.parse(
  fs.readFileSync(baseStructurePath, 'utf8'),
);

const extractPaths = (pages: BasePage[]): string[] => {
  const paths: string[] = [];

  for (const page of pages) {
    if (typeof page === 'string') {
      paths.push(page);
    } else {
      paths.push(...extractPaths(page.pages));
    }
  }

  return paths;
};

const pathToConstantName = (docPath: string): string => {
  return docPath.replace(/^\//, '').replace(/[/-]/g, '_').toUpperCase();
};

const allPaths: string[] = [];

for (const tab of baseStructure.tabs) {
  for (const group of tab.groups) {
    allPaths.push(...extractPaths(group.pages));
  }
}

const sortedPaths = [...allPaths].sort();

const AUTO_GENERATED_HEADER = `/*
 * _____                    _
 *|_   _|_      _____ _ __ | |_ _   _
 *  | | \\ \\ /\\ / / _ \\ '_ \\| __| | | | Auto-generated file
 *  | |  \\ V  V /  __/ | | | |_| |_| | DO NOT EDIT - changes will be overwritten
 *  |_|   \\_/\\_/ \\___|_| |_|\\__|\\__, | Generated by: yarn docs:generate-paths
 *                              |___/
 *
 * This file is generated from packages/twenty-docs/navigation/base-structure.json
 * To add new documentation pages, add them to base-structure.json and run:
 *   yarn docs:generate-paths
 */

`;

// Generate DocumentationPaths.ts
const pathEntries = sortedPaths
  .map((docPath) => {
    const constName = pathToConstantName(docPath);
    const value = `'/${docPath}'`;
    // Check if line would be too long (80 char limit)
    if (`  ${constName}: ${value},`.length > 80) {
      return `  ${constName}:\n    ${value},`;
    }
    return `  ${constName}: ${value},`;
  })
  .join('\n');

const pathsContent = `${AUTO_GENERATED_HEADER}export const DOCUMENTATION_PATHS = {
${pathEntries}
} as const;

export type DocumentationPath =
  (typeof DOCUMENTATION_PATHS)[keyof typeof DOCUMENTATION_PATHS];
`;

fs.writeFileSync(path.join(outputDir, 'DocumentationPaths.ts'), pathsContent);

console.log(`Generated documentation constants to ${outputDir}`);
console.log(`Total paths: ${sortedPaths.length}`);
